"""Employee management page."""

from __future__ import annotations

import datetime
from typing import Dict, Any

import streamlit as st
import pandas as pd

# Local imports
from services.employee_service import EmployeeService


def render() -> None:
    st.title("Employees")
    role = str(st.session_state.get("role", ""))  # "Admin" | "Manager" | "Staff"
    can_edit = role in ("Admin", "Manager")

    service = EmployeeService()

    # ----- Listing -----
    employees = service.list_employees(active_only=False)
    df = pd.DataFrame(employees) if employees else pd.DataFrame(
        columns=["id", "emp_code", "full_name", "designation", "email", "phone", "date_joined", "is_active"]
    )
    st.subheader("Employee List")
    st.dataframe(df, use_container_width=True)

    if not can_edit:
        st.info("You have view-only access. Contact Admin/Manager for changes.")
        return

    # ----- Add new employee -----
    st.subheader("Add Employee")
    with st.form("emp_add_form", clear_on_submit=True):
        emp_code = st.text_input("Employee Code*", max_chars=50)
        full_name = st.text_input("Full Name*", max_chars=120)
        designation = st.text_input("Designation", max_chars=80)
        email = st.text_input("Email", max_chars=120)
        phone = st.text_input("Phone", max_chars=30)
        date_joined = st.date_input("Date Joined", value=datetime.date.today(), format="YYYY-MM-DD")
        is_active = st.checkbox("Active", value=True)

        submitted_add = st.form_submit_button("Add Employee")

    if submitted_add:
        try:
            payload: Dict[str, Any] = {
                "emp_code": emp_code.strip(),
                "full_name": full_name.strip(),
                "designation": designation.strip(),
                "email": email.strip(),
                "phone": phone.strip(),
                "date_joined": date_joined,  # repo serializes to ISO
                "is_active": bool(is_active),
            }
            if not payload["emp_code"] or not payload["full_name"]:
                raise ValueError("Employee Code and Full Name are mandatory.")

            service.add_employee(payload)
            st.success("Employee added successfully.")
            st.rerun()  # NOTE: modern API (replaces deprecated st.experimental_rerun)
        except Exception as e:
            st.error(f"Unable to add employee: {e}")

    # ----- Edit existing employee -----
    st.subheader("Edit Employee")
    if df.empty:
        st.info("No employees to edit.")
        return

    # Choose employee to edit
    choices = {f"{row['emp_code']} — {row['full_name']}": int(row["id"]) for _, row in df.iterrows()}
    selected_label = st.selectbox("Select Employee", options=list(choices.keys()))
    selected_id = choices[selected_label]
    current = next((r for r in employees if r.get("id") == selected_id), None) or {}

    with st.form("emp_edit_form"):
        new_emp_code = st.text_input("Employee Code*", value=str(current.get("emp_code", "")), max_chars=50)
        new_full_name = st.text_input("Full Name*", value=str(current.get("full_name", "")), max_chars=120)
        new_designation = st.text_input("Designation", value=str(current.get("designation", "")), max_chars=80)
        new_email = st.text_input("Email", value=str(current.get("email", "")), max_chars=120)
        new_phone = st.text_input("Phone", value=str(current.get("phone", "")), max_chars=30)

        # Handle date_joined which may be ISO string in TinyDB
        joined_val = current.get("date_joined")
        if isinstance(joined_val, str):
            try:
                y, m, d = map(int, joined_val.split("-"))
                joined_default = datetime.date(y, m, d)
            except Exception:
                joined_default = datetime.date.today()
        elif isinstance(joined_val, datetime.date):
            joined_default = joined_val
        else:
            joined_default = datetime.date.today()

        new_date_joined = st.date_input("Date Joined", value=joined_default, format="YYYY-MM-DD")
        new_is_active = st.checkbox("Active", value=bool(current.get("is_active", True)))

        col1, col2 = st.columns(2)
        submitted_update = col1.form_submit_button("Update")
        submitted_delete = col2.form_submit_button("Delete", type="secondary")

    if submitted_update:
        try:
            if not new_emp_code.strip() or not new_full_name.strip():
                raise ValueError("Employee Code and Full Name are mandatory.")
            updated = service.update_employee(
                selected_id,
                emp_code=new_emp_code.strip(),
                full_name=new_full_name.strip(),
                designation=new_designation.strip(),
                email=new_email.strip(),
                phone=new_phone.strip(),
                date_joined=new_date_joined,  # repo serializes to ISO
                is_active=bool(new_is_active),
            )
            if updated:
                st.success("Employee updated successfully.")
                st.rerun()
            else:
                st.warning("No changes were applied.")
        except Exception as e:
            st.error(f"Unable to update employee: {e}")

    if submitted_delete:
        try:
            if service.delete_employee(selected_id):
                st.success("Employee deleted.")
                st.rerun()
            else:
                st.warning("Delete failed or employee not found.")
        except Exception as e:
            st.error(f"Unable to delete employee: {e}")
